"use client"

import { useMemo, useState } from "react";
import { FadeIn } from "@/components/animations/fade-in";
import { Loader, Check, X, Copy, } from "lucide-react";
import { useStatusLoadingMessages } from "../hook/use-status-loading-messages";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { Button } from "@/components/ui/button";
import { cn } from "@/lib/utils";
import { focusRing } from "@/lib/styles";
import { potentialDateToDateStringFormatter } from "@/lib/formatters/date-formatter";
import { useStatusLoadingContentForAgency, UseStatusLoadingContentForAgencyProps } from "../hook/use-status-loading-content";
import { AppValidationError } from "@/zod/feeds.schema";
import { StatusBannerContentWrapper, StatusBannerIconWrapper, StatusBannerWrapper } from "./status.shared";

export function StatusRevalidateBannerForAgency({ agency, delay }: { agency: string, delay?: number }) {
  const content = useMemo(() => {
    return {
      agency: agency,
      loading: <INTERNAL__Loading />,
      success: {
        content: <INTERNAL__Success date={new Date(Date.now())} />,
        timeout: 10000,
      },
      error: (_s, e) => <INTERNAL__Error error={e} />,
    } satisfies UseStatusLoadingContentForAgencyProps
  }, [agency])

  const {
    isLoading,
    data: showing
  } = useStatusLoadingContentForAgency(content)

  if (isLoading || !showing) {
    return
  }

  return (
    <FadeIn delay={delay}>
      <StatusBannerWrapper>
        {showing}
      </StatusBannerWrapper>
    </FadeIn>
  )
}

function INTERNAL__Loading() {
  const message = useStatusLoadingMessages()
  return (
    <>
      <StatusBannerIconWrapper className="animate-pulse">
        <Loader className="opacity-60 animate-spin size-4 stroke-2" />
      </StatusBannerIconWrapper>
      <StatusBannerContentWrapper className="animate-pulse">
        <p className="text-sm font-medium">A build is currently in progress.</p>
        <p className="text-xs text-muted-foreground">{message}</p>
      </StatusBannerContentWrapper>
    </>
  )
}

function INTERNAL__Success({ date }: { date: Date }) {
  return (
    <>
      <StatusBannerIconWrapper>
        <Check className="opacity-60 size-4 stroke-2 text-green-500 dark:text-green-600" />
      </StatusBannerIconWrapper>
      <StatusBannerContentWrapper>
        <p className="text-sm font-medium">Open trip planner server was successfully built!</p>
        <p className="text-xs text-muted-foreground">Successfully built at {potentialDateToDateStringFormatter(date)}</p>
      </StatusBannerContentWrapper>
    </>
  )
}

function INTERNAL__Error({
  error,
}: {
  error: AppValidationError
}) {
  const [copied, setCopied] = useState<boolean>(false);

  const handleOpen = async () => {
    await navigator.clipboard.writeText(error ?? "")
    setCopied(true);
    setTimeout(() => setCopied(false), 1500);
  };

  return (
    <>
      <StatusBannerIconWrapper>
        <X className="opacity-60 size-4 stroke-2 text-destructive" />
      </StatusBannerIconWrapper>

      <StatusBannerContentWrapper>
        <p className="text-sm font-medium">Error occurred during build.</p>
        <p className="text-xs text-muted-foreground line-clamp-3">{error}</p>
      </StatusBannerContentWrapper>

      <TooltipProvider delayDuration={0}>
        <Tooltip>
          <TooltipTrigger asChild>
            <Button
              onClick={handleOpen}
              variant={"outline"}
              size="icon"
              className={cn(
                focusRing,
                "inline-flex items-center justify-center whitespace-nowrap text-sm font-medium disabled:pointer-events-none disabled:opacity-50 border border-transparent !outline-none focus-visible:!z-[100] transition-shadow",
                "relative col-span-1 bg-transparent rounded-none",
                "aspect-video border-s border-s-border border-t border-t-border rounded-ee-xl focus-visible:!border-s focus-visible:!border-e",
                "h-full w-16 md:w-16 md:aspect-square md:rounded-ee-none md:rounded-e-xl md:border-s md:border-s-border md:border-t-transparent",
                "group focus-visible:!text-primary focus-visible:!border-primary ring-primary/20 "
              )}
              aria-label={copied ? "Copied" : "Copy to clipboard"}
              disabled={copied}
            >
              <div
                className={cn(
                  "transition-all",
                  copied ? "scale-100 opacity-100" : "scale-0 opacity-0",
                )}
              >
                <Check
                  className="stroke-emerald-500 size-4"
                  aria-hidden="true"
                />
              </div>

              <div className={"sr-only"}>
                Copy error message to clipboard
              </div>

              <div
                className={cn(
                  "absolute transition-all",
                  copied ? "scale-0 opacity-0" : "scale-100 opacity-100",
                )}
              >
                <Copy className="group-focus-visible:text-destructive text-muted-foreground size-3" aria-hidden="true" />
              </div>
            </Button>
          </TooltipTrigger>
          <TooltipContent className="border border-input bg-popover px-2 py-1 text-xs text-muted-foreground">
            Copy error message to clipboard
          </TooltipContent>
        </Tooltip>
      </TooltipProvider>
    </>
  )
}
